//*************************************************************************************************
//
//	Description:
//		additive.fx - Simple shader for rendering simple additive translucent shapes, such as light cones
//									etc. Don't use for anything non-convex.
//
//	<P> Copyright (c) 2006 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     26/06/2008  0.1           Forked from basic.fx
//	<TABLE>
//
//*************************************************************************************************

#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//


// Compiler test settings, exercises all options
#if defined( TEST_COMPILE )
#endif



//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//


//
// Transforms
//
#if defined( _3DSMAX_ ) || defined(_PS3_)
// Max doesn't support viewproj as an app-supplied parameter
SHARE_PARAM float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;



//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// Vertex alpha channel (max presents it seperately for no good reason)
int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = -2;
	bool ColorChannel = true;
	bool export = false;
>;

#endif


//
// Textures
//

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Tex {UV1}";
	bool appEdit = true;
>;


float brightness
<
	string UIName = "Brightness";
	string UIWidget = "Spinner";
	float UIMin = 0.0f;
	float UIMax = 16.0f;
	bool appEdit = true;
	bool export = true;
> = 1.0f;


//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS



//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};


//
// Functions
//


//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position : POSITION;														// Object space position
#ifdef _3DSMAX_
	float3 colour   : TEXCOORD1;													// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
	float3 alpha		: TEXCOORD2;													// Vertex alpha
#else
	float4 colour   : COLOR0;															// Vertex colour
	float2 texCoord : TEXCOORD0;													// UV channel 1 texture coord
#endif
};


// Output structure
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD1;														// Vertex colour
	float2 texCoord		: TEXCOORD0;												// UV coords for texture channel 0
};



//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT AdditiveVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(_PS3_)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour * brightness;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
	_output.colour.rgb *= brightness;
#endif
	_output.texCoord = _input.texCoord;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

// Input structure

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

struct PSINPUT
{
	float4 colour			: TEXCOORD1;								// Vertex colour
	float2 texCoord		: TEXCOORD0;						// UV coords for texture channel 0
};

#else

// Input structure
struct PSINPUT
{
	float4 colour			: TEXCOORD1;											// Vertex colour
	float2 texCoord		: TEXCOORD0_centroid;					// UV coords for texture channel 0
};

#endif	// defined( _3DSMAX_ )


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT AdditiveFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	// Read textures
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	// Apply interpolated colour & global brightness
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	// Multiply by per-pixel brightness boost (from alpha)
	accumulatedColour.rgb *= diffuseTexColour.a;
	accumulatedColour.w = _input.colour.w;

	_output.Colour = CalculateOutputPixel( accumulatedColour );

	return _output;
}


//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Additive
<
	bool supportsSpecialisedLighting = false;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Additive";
	int    normalDeferredID		= 2;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "Additive";
	int    lowDetailDeferredID = 2;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool	ZEnable = true;
		bool	ZWriteEnable = false;
		bool	AlphaBlendEnable = true;
		string SrcBlend = "SRCALPHA";
		string DestBlend = "ONE";
		string BlendOp = "ADD";
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = true;
		SrcBlend = SRCALPHA;
		DestBlend = ONE;
		BlendOp = ADD;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx AdditiveVertexShader();
		PixelShader = compile sce_fp_rsx AdditiveFragmentShader();
#else
		VertexShader = compile vs_3_0 AdditiveVertexShader();
		PixelShader = compile ps_3_0 AdditiveFragmentShader();
#endif
	}
}
